	function [G, Z, P] = MULTIPLE_NOTCH_IIR(wTnotch, BWnotch, Norder)
	
	%	Computes the poles and zero of an IIR filter of order Norder 
	% 	with  multiple notches, given in wTnotch, each with the 3-dB 
	%	bandwidth, given in BWnotch.
	% 	The filter consists of an even-order allpass filter in parallel 
	%	with a unit branch.
	
	%	Toolbox for DIGITAL FILTERS USING MATLAB
	
	% 	Modified by: 	 	Lars Wanhammar 2004-07-29
	% 	Version:			1 
	% 	Known bugs:			 
	% 	Report bugs to:		Wanhammar@gmail.com
	
	%	Reference
	% 	Pei S.-C. and Tseng C.-C.: IIR multiple notch filter 
	%	design based on allpass filters, IEEE Trans. Signal 
	%	Processing, Part II, Vol. 44, No. 2, pp. 133-136, Feb. 1997.
	
	if mod(Norder, 2) == 1
		display(['must be an even order filter'])
	end
	if length(wTnotch) ~= length(BWnotch)
		disp([' error in wTnotch or BWnotch'])
	end
	M = Norder/2;
	wn = zeros(1,2*M); BW = zeros(1,2*M);
	Q = zeros(2*M);
	for n = 1:2*M
		x = floor((n+1)/2);
		w(n) = wTnotch(x)-0.5*(1 -(-1)^mod(n,2))*BWnotch(x)/2;
		Fi(n) = -(2*x-1)*pi + 0.5*(1 -(-1)^mod(n,2))*pi/2;
		b(n) = 0.5*(Fi(n)+2*M*w(n));
	end
	p = tan(b)';
	for n = 1:2*M
		for k = 1:2*M
			Q(n,k) = sin(k*w(n))-tan(b(n))*cos(k*w(n));
		end
	end		
	a = Q\p;
	% 2*M-order allpass filter
	Den = [1,a'];
	Num = fliplr(Den);
	% Transfer function
	G = sum(Den)/sum(Num);
	Z = roots(Den+Num);
	P = roots(Den);
	
